/*
* prof.c- Sigmastar
*
* Copyright (c) [2019~2020] SigmaStar Technology.
*
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License version 2 for more details.
*
*/
#include <linux/kernel.h>
#include <linux/types.h>
#include "ms_platform.h"
#include "ms_types.h"
#include <linux/io.h>

#define RECORD_IPL_ADDR         0xF900F000
#define RECORD_BL_ADDR          0xF900F400
#define RECORD_LINUX_ADDR       0xF900F800  // 0xF900F800 ~ 0xF9018000

#define MAX_IPL_RECORD 31
#define MAX_IPL_LENGTH 24

struct timestamp_ipl {
    unsigned int timestamp_us;          /* 4                  */
    unsigned int mark;                  /* 4,  ex:__line__    */
    unsigned char name[MAX_IPL_LENGTH]; /* 24, ex:__function__*/
};

struct timerecord_ipl {
    U32 count;
    struct timestamp_ipl tt[MAX_IPL_RECORD];
};

#define MAX_BL_RECORD 31
#define MAX_BL_LENGTH 24

struct timestamp_bl {
    unsigned int timestamp_us;          /* 4                  */
    unsigned int mark;                  /* 4,  ex:__line__    */
    unsigned char name[MAX_BL_LENGTH];  /* 24, ex:__function__*/
};

struct timerecord_bl {
    U32 count;
    struct timestamp_bl tt[MAX_BL_RECORD];
};

#define MAX_LINUX_RECORD 100                /*max:(0x18000-0xF800)/40~=870*/
#define MAX_LINUX_LENGTH 32

struct timestamp_linux {
    unsigned int timestamp_us;              /* 4                  */
    unsigned int mark;                      /* 4,  ex:__line__    */
    unsigned char name[MAX_LINUX_LENGTH];   /* 32, ex:__function__*/
};

struct timerecord_linux {
    U32 count;
    struct timestamp_linux tt[MAX_LINUX_RECORD];
};

int g_record_inited = 0;
void* g_addr_record_ipl = 0;
void* g_addr_record_bl = 0;
void* g_addr_record_kernel= 0;

void record_timestamp_init(void)
{
    g_record_inited = 0;
}

U64 arch_counter_get_cntpct(void)
{
    U64 cval;
    asm volatile("isb " : : : "memory");
    asm volatile("mrrc p15, 0, %Q0, %R0, c14" : "=r" (cval));
    return cval;
}

unsigned int read_timestamp(void)
{
    u64 cycles=arch_counter_get_cntpct();
    u64 usticks=div64_u64(cycles, 6/*us_ticks_factor*/);
    return (unsigned int) usticks;
}

void record_timestamp(int mark, const char *name)
{
    struct timerecord_linux *tc;

    tc = (struct timerecord_linux *) (RECORD_LINUX_ADDR);
    if(!g_record_inited)
    {
        tc->count = 0;
        g_record_inited=1;
    }

    if (tc->count < MAX_LINUX_RECORD)
    {
        tc->tt[tc->count].timestamp_us = read_timestamp();
        tc->tt[tc->count].mark = mark;
        strncpy(tc->tt[tc->count].name, name, MAX_LINUX_LENGTH - 1);
        tc->count++;
    }
}
EXPORT_SYMBOL(record_timestamp);

void record_timestamp_ext(int mark, const char *name, unsigned int timestamp)
{
    struct timerecord_linux *tc;

    tc = (struct timerecord_linux *) (RECORD_LINUX_ADDR);
    if(!g_record_inited)
    {
        tc->count = 0;
        g_record_inited=1;
    }

    if (tc->count < MAX_LINUX_RECORD)
    {
        tc->tt[tc->count].timestamp_us = timestamp;
        tc->tt[tc->count].mark = mark;
        strncpy(tc->tt[tc->count].name, name, MAX_LINUX_LENGTH - 1);
        tc->count++;
    }
}

/* returns number of characters written to buf, excluding terminating \0 */
int record_show(char *buf, int buf_len)
{
    struct timerecord_ipl *tc_ipl;
    struct timerecord_bl *tc_bl;
    struct timerecord_linux *tc_linux;
    int i=0;
    int write_len = 0;

    if (buf == NULL || buf_len == 0) {
        printk(KERN_ERR"No buffer to store timestamp data");
        return 0;
    }

    tc_ipl = (struct timerecord_ipl *) (RECORD_IPL_ADDR); // IMI SRAM
    if(tc_ipl->count <= MAX_IPL_RECORD && tc_ipl->count > 0)
    {
        write_len += scnprintf(buf + write_len, buf_len - write_len, "IPL: 0x%p\n", tc_ipl);
        for(i = 0; i < tc_ipl->count; i++)
        {
            tc_ipl->tt[i].name[MAX_IPL_LENGTH-1] = '\0';

            write_len += scnprintf(buf + write_len, buf_len - write_len, "%02d st:%8u, diff:%8u, %s, %d\n",
                i,
                tc_ipl->tt[i].timestamp_us,
                tc_ipl->tt[i].timestamp_us - tc_ipl->tt[i ? i-1 : i].timestamp_us,
                tc_ipl->tt[i].name,
                tc_ipl->tt[i].mark);
        }
        write_len += scnprintf(buf + write_len, buf_len - write_len, "Total cost:%8u(us)\n",  tc_ipl->tt[tc_ipl->count - 1].timestamp_us - tc_ipl->tt[0].timestamp_us);
    }

    Chip_Inv_Cache_Range(RECORD_BL_ADDR, RECORD_LINUX_ADDR - RECORD_BL_ADDR);
    tc_bl = (struct timerecord_bl *) (RECORD_BL_ADDR); // IMI SRAM
    if(tc_bl->count <= MAX_BL_RECORD && tc_bl->count > 0)
    {
        write_len += scnprintf(buf + write_len, buf_len - write_len, "BL: 0x%p\n", tc_bl);
        for(i = 0; i < tc_bl->count; i++)
        {
            tc_bl->tt[i].name[MAX_BL_LENGTH - 1] = '\0';

            write_len += scnprintf(buf + write_len, buf_len - write_len, "%02d st:%8u, diff:%8u, %s, %d\n",
                i,
                tc_bl->tt[i].timestamp_us,
                tc_bl->tt[i].timestamp_us - tc_bl->tt[i ? i - 1 : i].timestamp_us,
                tc_bl->tt[i].name,
                tc_bl->tt[i].mark);
        }
        write_len += scnprintf(buf + write_len, buf_len - write_len, "Total cost:%8u(us)\n", tc_bl->tt[tc_bl->count - 1].timestamp_us - tc_bl->tt[0].timestamp_us);
    }

    tc_linux = (struct timerecord_linux *) (RECORD_LINUX_ADDR); // IMI SRAM
    if(tc_linux->count <= MAX_LINUX_RECORD && tc_linux->count > 0)
    {
        write_len += scnprintf(buf + write_len, buf_len - write_len, "Linux:0x%p\n", tc_linux);
        for(i = 0; i < tc_linux->count; i++)
        {
            tc_linux->tt[i].name[MAX_LINUX_LENGTH - 1] = '\0';

            write_len += scnprintf(buf + write_len, buf_len - write_len, "%02d st:%8u, diff:%8u, %s, %d\n",
                i,
                tc_linux->tt[i].timestamp_us,
                tc_linux->tt[i].timestamp_us - tc_linux->tt[i ? i - 1 : i].timestamp_us,
                tc_linux->tt[i].name,
                tc_linux->tt[i].mark);
        }
        write_len += scnprintf(buf + write_len, buf_len - write_len, "Total cost:%8u(us)\n", tc_linux->tt[tc_linux->count - 1].timestamp_us - tc_linux->tt[0].timestamp_us);
    }
    return write_len;
}

